<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use App\Services\CrudStubGenerator;

class MakeCrudCommand extends Command
{
    protected $signature = 'make:crud {name} {folder}';
    protected $description = 'Create CRUD files including model, migration, controller, request, response, and repository';

    protected CrudStubGenerator $generator;

    // Inject the generator via constructor
    public function __construct(CrudStubGenerator $generator)
    {
        parent::__construct();
        $this->generator = $generator;
    }

    public function handle()
    {
        $name = Str::studly($this->argument('name'));
        $folder = Str::studly($this->argument('folder'));

        $this->makeModel($name, $folder);
        $this->makeController($name, $folder);
        $this->makeRequest($name, $folder);
        $this->makeResponse($name, $folder);
        $this->makeRepository($name, $folder);

        $this->info("✅ CRUD for $name created under folder: $folder.");
    }

    protected function makeModel($name, $folder)
    {
        Artisan::call('make:model', [
            'name' => "$folder/$name",
            '-m' => true,
        ]);

        $path = app_path("Models/$folder/$name.php");
        File::put($path, $this->generator->getModelContent($name, $folder));

        $this->info("✔ Model & migration for $name created.");
    }

    protected function makeController($name, $folder)
    {
        $path = app_path("Http/Controllers/Api/$folder/{$name}Controller.php");
        File::ensureDirectoryExists(dirname($path));

        File::put($path, $this->generator->getControllerContent($name, $folder));

        $this->info("✔ Controller for $name created.");
    }

    protected function makeRequest($name, $folder)
    {
        $path = app_path("Http/Requests/$folder/{$name}Request.php");
        File::ensureDirectoryExists(dirname($path));

        File::put($path, $this->generator->getRequestContent($name, $folder));

        $this->info("✔ Request for $name created.");
    }

    protected function makeResponse($name, $folder)
    {
        $path = app_path("Http/Responses/$folder/{$name}Response.php");
        File::ensureDirectoryExists(dirname($path));

        File::put($path, $this->generator->getResponseContent($name, $folder));

        $this->info("✔ Response for $name created.");
    }

    protected function makeRepository($name, $folder)
    {
        $path = app_path("Repositories/$folder/{$name}Repository.php");
        File::ensureDirectoryExists(dirname($path));

        File::put($path, $this->generator->getRepositoryContent($name, $folder));

        $this->info("✔ Repository for $name created.");
    }
}
