<?php

namespace App\Repositories\Main;

use App\Models\Main\User;
use App\Models\Main\UserPermission;
use Illuminate\Support\Facades\Hash;

class UserRepository
{
    /**
     * Get paginated list of users
     *
     * @param int $rows
     * @return array
     */
    public function index(array $params = [])
    {
        $query = User::query();

        // Apply search filter if $search parameter is provided
        if (isset($params['word']) && !empty($params['word'])) {
            $query->where(function ($q) use ($params) {
                $q->where('name', 'LIKE', "%{$params['word']}%")
                ->orWhere('email', 'LIKE', "%{$params['word']}%")
                  ->orWhere('username', 'LIKE', "%{$params['word']}%");

            });
        }
        if(isset($params['is_active'])) {
            $query->where('is_active', $params['is_active']);
        }

        if(isset($params['perPage']) && $params['perPage'] > 0) {
            $users = $query->paginate($params['perPage'], ['*'], 'page', $params['page'] ?? 1);
        } else {
            $users = $query->get();
        }

        return $users;
    }

    /**
     * Show single user by ID
     *
     * @param int $id
     * @return User|null
     */
    public function show(int $id): ?User
    {
        return User::where('id', $id)
            ->first();
    }

    /**
     * Create new user
     *
     * @param array $data
     * @return User
     */
    public function create(array $data): User
    {
        $data['password'] = Hash::make($data['password']);

        return User::create($data);
    }

    /**
     * Update existing user by ID
     *
     * @param int $id
     * @param array $data
     * @return User|null
     */
    public function update(int $id, array $data): ?User
    {
        $user = User::where('id', $id)
            ->first();

        if (!$user) {
            return null;
        }

        if (isset($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        }

        $user->update($data);

        return $user;
    }
    /**
     * Soft delete user by ID
     *
     * @param int $id
     * @return bool
     */
    public function delete(int $id): bool
    {
        $user = User::where('id', $id)
            ->where('is_active', true)
            ->first();

        if (!$user) {
            return false;
        }

        $user->update(['is_active' => true]);

        return true;
    }
}
