<?php

namespace App\Http\Controllers\API\V1\Assets;

use App\Classes\Assets\AssetsHelper;
use App\Classes\General\Upload;
use App\Http\Controllers\API\V1\BaseController;
use Illuminate\Http\Request;
use App\Models\Main\ScreenSub;
use App\Models\Assets\Assets;
use App\Models\Assets\AssetsCategories;
use App\Models\Finance\AccountTree;
use App\Models\Finance\CostCenter;
use App\Models\Finance\Tax;
use App\Models\Finance\Treasury;
use App\Models\Main\Bank;
use App\Models\Main\Locations;
use App\Models\Purchases\PurchasesVendor;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rule;

class AssetsController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang , Request $request)
    {
        $this->checkPermission('06-021','view');
        $this->filter = array(
            'lang'        => $lang ,
            'active'      => $request['active'] ?? 1 ,
            'rows'        => $request['rows'] ?? 15 ,
            'page'        => $request['page'] ?? 1 ,
            'word'        => $request['word'] ?? null
        );

        $this->return['screen_name']    = ScreenSub::get_screen_sub_by_screen_code('06-021' , $lang)->sub_title ?? '';
        $this->return['items']          = Assets::all_assets_m($this->filter);
        $this->return['assets_categories'] = AssetsCategories::all_assets_category(array('active' => 1, 'lang' => $lang));
        return $this->sendResponse($this->return );
    }

    public function create($lang)
    {
        $this->checkPermission('06-021','add');

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-021' , $lang)->sub_title ?? '';
        $this->return['cost_centers']      = CostCenter::active_cost_centers( $lang , [ 'active' => 1 , 'account_type' => 2 ] );
        $this->return['purchases_vendors'] = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
        $this->return['assets_categories'] = AssetsCategories::all_assets_category(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(['lang' => $lang , 'active' => 1 , 'type'=>2 ]);
        $this->return['account_trees']     = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));
        $this->return['locations']         = Locations::get_all(['lang' => $lang , 'active' => 1 ]);
        $this->return['banks']             = Bank::all_banks(array('lang' => $lang, 'active' => 1 ));
        $this->return['treasurys']         = Treasury::all_treasury(['lang' => $lang, 'active' => 1, 'user' => auth()->user()->id]);
        return $this->sendResponse( $this->return);
    }

    public function store(Request $request)
    {
        $this->checkPermission('06-021','add');

        $input    = $request->all();
        $category = AssetsCategories::get_assets_category($input['asset_category_id']);
        $year   = array_search(date("Y", strtotime($input['purchase_date'])), $this->data['years']);
        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }
        $codes = (new AssetsHelper)->get_next_codes($input['purchase_date'], $year, $this->data['current_branch']);
        $asset_code = (new AssetsHelper)->generate_code($input['asset_category_id']);
        $insert = [
            'branch_id'                 => $this->data['current_branch'],
            'finance_year_id'           => $year,
            'year_ser'                  => $codes['year_ser'],
            'month_ser'                 => $codes['month_ser'],
            'branch_ser'                => $codes['branch_ser'],
            'daily_journal_m_id'        => 0,
            'name_en'                   => $input['name_en'],
            'name_ar'                   => $input['name_ar'],
            'description'               => $input['description'] ?? null,
            'asset_category_id'         => $input['asset_category_id'] ?? 0,
            'asset_code'                => $asset_code,
            'total_vat'                 => $input['total_vat'] ?? 0,
            'total_value_with_vat'      => $input['total_value_with_vat'] ?? 0,
            'total_value_without_vat'   => $input['total_value_without_vat'] ?? 0,
            'annual_depreciation'       => $input['annual_depreciation'] ?? ($input['total_value_without_vat']/($category->depr_months/12)),
            'daily_depreciation'        => $input['daily_depreciation'] ?? (($input['total_value_without_vat']/($category->depr_months/12)) / 365),
            'opening_value'             => $input['opening_value'] ?? 0,
            'dev_value'                 => $input['dev_value'] ?? 0,
            'depr_value'                => $input['depr_value'] ?? 0,
            'remain_value'              => $input['total_value_without_vat'] -  $input['opening_value'] ?? 0,
            'last_month_depreciation'   => $input['last_month_depreciation'] ?? null,
            'location_id'               => $input['location_id'] ?? 0,
            'cost_center_id'            => $input['cost_center_id'] ?? 0,
            'asset_acc_id'              => $input['asset_acc_id'] ?? 0,
            'destruction_calculate'     => $input['destruction_calculate'] ?? 0,
            'stop_dealing'              => $input['stop_dealing'] ?? 0,
            'previous_years'            => $input['previous_years'] ?? 0,
            'purchase_date'             => $input['purchase_date'] ?? date('Y-m-d'),
            'depreciation_type'         => $input['depreciation_type'] ?? null,
            'depreciation_method'       => $input['depreciation_method'] ?? null,
            'total_depr_acc_id'         => $input['total_depr_acc_id'] ?? 0,
            'profit_loss_acc_id'        => $input['profit_loss_acc_id'],
            'dev_spare_acc_id'          => $input['dev_spare_acc_id'],
            'rev_spare_acc_id'          => $input['rev_spare_acc_id'],
            'exp_depr_acc_id'           => $input['exp_depr_acc_id'] ?? 0,
            'status'                    => $input['status'] ?? 1,
            'add_user'                  => auth()->user()->id,
        ];
        $validator = validator()->make( $insert , Assets::$rules );
        if($validator->fails())
        {
            return $this->sendError( trans('error.add_fails') , $validator->errors() );
        }

        DB::beginTransaction();

        $this->return['id'] = Assets::add_assets_m($insert);

        if(!isset($input['payments'])){
            return $this->sendError(trans('error.no_payments'));
        }else{
            foreach($input['payments'] as $payment){
            $payment_insert = [
                'asset_m_id'        => $this->return['id'],
                'description'       => $payment['description'] ?? null,
                'value_without_vat' => $payment['value_without_vat'],
                'value_with_vat'    => $payment['value_with_vat'],
                "vat"               => $payment['vat'],
                'tax_id'            => $payment['tax_id'] ?? 0,
                'payment_method'    => $payment['payment_method'] ?? 1,
                'payment_method_id' => $payment['payment_method_id'] ?? 1,


            ];
            Assets::add_assets_d($payment_insert);
            }
        }
        if($insert['previous_years'] == 0){
            (new AssetsHelper)->generate_journal($this->return['id']);
        }

        DB::commit();

        return $this->sendResponse( $this->return , trans('main.add_success') );
    }

    public function show($lang , $id )
    {
        $this->checkPermission('06-021','view');

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-021' , $lang)->sub_title ?? '';
        $this->return['cost_centers']      = CostCenter::active_cost_centers( $lang , [ 'active' => 1 , 'account_type' => 2 ] );
        $this->return['purchases_vendors'] = PurchasesVendor::all_purchases_vendors(array('active' => 1, 'lang' => $lang));
        $this->return['assets_categories'] = AssetsCategories::all_assets_category(array('active' => 1, 'lang' => $lang));
        $this->return['account_trees']     = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));
        $this->return['taxes']             = Tax::all_taxes(['lang' => $lang , 'active' => 1  , 'type' => 2]);
        $this->return['item']              = Assets::get_assets_m($id, $lang);
        $this->return['attachments']       = Assets::get_assets_attachments($id,$lang);
        $this->return['payments']          = Assets::get_assets_payments($id, $lang);
        $this->return['locations']         = Locations::get_all(['lang' => $lang , 'active' => 1 ]);
        $this->return['banks']             = Bank::all_banks(array('lang' => $lang, 'active' => 1 ));
        $this->return['treasurys']         = Treasury::all_treasury(['lang' => $lang, 'active' => 1, 'user' => auth()->user()->id]);

        return $this->sendResponse( $this->return , '' );
    }

    public function update($lang, $id , Request $request)
    {
        $this->checkPermission('06-021','edit');
        if(!(new AssetsHelper)->check_status($id , $this->data['current_year'])){
            return $this->sendError(trans('error.asset_allredy_has_deprecated_or_not_same_year'));
        };
        $input  = $request->all();
        $category = AssetsCategories::get_assets_category($input['asset_category_id']);
        $update = [
            'name_en'                   => $input['name_en'],
            'name_ar'                   => $input['name_ar'],
            'description'               => $input['description'] ?? null,
            'asset_category_id'         => $input['asset_category_id'] ?? 0,
            'total_vat'                 => $input['total_vat'] ?? 0,
            'total_value_with_vat'      => $input['total_value_with_vat'] ?? 0,
            'total_value_without_vat'   => $input['total_value_without_vat'] ?? 0,
            'annual_depreciation'       => $input['annual_depreciation'] ?? ($input['total_value_without_vat']/($category->depr_months/12)),
            'daily_depreciation'        => $input['daily_depreciation'] ?? (($input['total_value_without_vat']/($category->depr_months/12)) / 365),
            'opening_value'             => $input['opening_value'] ?? 0,
            'dev_value'                 => $input['dev_value'] ?? 0,
            'depr_value'                => $input['depr_value'] ?? 0,
            'remain_value'              => $input['total_value_without_vat'] -  $input['opening_value'] ?? 0,
            'last_month_depreciation'   => $input['last_month_depreciation'] ?? null,
            'location_id'               => $input['location_id'] ?? 0,
            'cost_center_id'            => $input['cost_center_id'] ?? 0,
            'asset_acc_id'              => $input['asset_acc_id'] ?? 0,
            'destruction_calculate'     => $input['destruction_calculate'] ?? 0,
            'stop_dealing'              => $input['stop_dealing'] ?? 0,
            'previous_years'            => $input['previous_years'] ?? 0,
            'purchase_date'             => $input['purchase_date'] ?? date('Y-m-d'),
            'depreciation_type'         => $input['depreciation_type'] ?? null,
            'depreciation_method'       => $input['depreciation_method'] ?? null,
            'total_depr_acc_id'         => $input['total_depr_acc_id'] ?? 0,
            'profit_loss_acc_id'        => $input['profit_loss_acc_id'],
            'dev_spare_acc_id'          => $input['dev_spare_acc_id'],
            'rev_spare_acc_id'          => $input['rev_spare_acc_id'],
            'exp_depr_acc_id'           => $input['exp_depr_acc_id'] ?? 0,
            'status'                    => $input['status'] ?? 1,
            'add_user'                  => auth()->user()->id,
        ];
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('assets_m')->ignore($id,'asset_m_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('assets_m')->ignore($id,'asset_m_id'),
            ],
        ];

        $validator  = validator()->make( $update , $update_rules );

        if($validator->fails())
        {
            return $this->sendError( trans('error.add_fails') , $validator->errors() );
        }

        DB::beginTransaction();

            Assets::update_assets_m($update ,$id);

            if(!isset($input['payments'])){
                return $this->sendError(trans('error.no_payments'));
            }else{
                DB::table('assets_purchases')->where('asset_m_id',$id)->delete();
                foreach($input['payments'] as $payment){
                $payment_insert = [
                    'asset_m_id'        => $id,
                    'description'       => $payment['description'] ?? null,
                    'value_without_vat' => $payment['value_without_vat'],
                    'value_with_vat'    => $payment['value_with_vat'],
                    "vat"               => $payment['vat'],
                    'tax_id'            => $payment['tax_id'] ?? 0,
                    'payment_method'    => $payment['payment_method'] ?? 1,
                    'payment_method_id' => $payment['payment_method_id'] ?? 1,

                ];
                Assets::add_assets_d($payment_insert);
                }
            }
            // if($update['previous_years'] == 0){
            //     (new AssetsHelper)->generate_journal($id);
            // }
        DB::commit();

        $this->return = Assets::get_assets_m($id,$lang);
        return $this->sendResponse( $this->return , trans('main.update_success') );
    }

    public function destroy($lang,$id)
    {
        $this->checkPermission('06-021','delete');
        if(!(new AssetsHelper)->check_status($id)){
            return $this->sendError(trans('error.asset_allredy_has_deprecated_or_not_same_year'));
        };
        DB::beginTransaction();
        Assets::delete_assets_m($id);
        (new AssetsHelper)->generate_journal($id);
        DB::commit();
        return $this->sendResponse( [] , trans('main.delete_success') );
    }


    public function toggle_active($lang,$id){

        $item = Assets::get_assets_m($id,$lang);
        if($item->active){

            $this->checkPermission('06-021','delete');
            if(!(new AssetsHelper)->check_status($id)){
                return $this->sendError(trans('error.asset_allredy_has_deprecated'));
            };
            DB::beginTransaction();
            Assets::delete_assets_m($id);
            (new AssetsHelper)->generate_journal($id);
            DB::commit();
            $this->return['active'] = false;

        }else{

            $this->checkPermission('06-021','edit');

            Assets::update_assets_m(array('active' => 1 ) , $id);
            $this->return['active'] = true;
        }
        return $this->sendResponse( $this->return , trans('main.update_success') );
    }
    public function add_attachments(Request $request)
    {
        $this->checkPermission('06-021','add');

        $input    = $request->all();

        $upload = (new Upload)->uploadFile($input['file_url'], 'assets_attachments', $input['file_name']);
                if(!$upload)
                    return $this->sendError(trans("error.Something Error"));
                    $this->return['id']=DB::table('assets_attachments')->insert([
                    'asset_m_id'        => $input['asset_m_id'],
                    'file_name'         => $upload['filename'],
                    'file_url'          => $upload['url'],
                    'add_user'          => auth()->user()->id,
                ]);

        return $this->sendResponse( $this->return , trans('main.add_success') );
    }
    public function delete_attachments($lang , $id)
    {
        $this->checkPermission('06-021','delete');

        $file_url = DB::table('assets_attachments')->where('attachment_id',$id)->value('file_url');
        //delete the file from the server
        if ($file_url && File::exists(public_path($file_url))) {
            File::delete(public_path($file_url));
        }
        Assets::delete_attachments($id);
        return $this->sendResponse( $this->return , trans('main.delete_success') );
    }


}
