<?php

namespace App\Http\Controllers\API\V1\Finance;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Finance\Tax;
use Illuminate\Http\Request;
use App\Classes\Finance\TaxHelper;
use App\Models\Finance\AccountTree;
use Illuminate\Validation\Rule;

use App\Models\Main\ScreenSub;

use DB;
class TaxController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('03-003','view');

        $this->filter = array(
            'lang'   => $lang ,
            'active' => $request['active'] ?? 1 ,
            'list'   => $request['list'] ?? null ,
            'rows'   => $request['rows'] ?? null ,
            'page'   => $request['page'] ?? 1 ,
            'word'   => $request['word'] ?? null ,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-003', $lang)->sub_title ?? '';
        $this->return['items']       = Tax::all_taxes($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('03-003','add');

        $this->return['screen_name']    = ScreenSub::get_screen_sub_by_screen_code('03-003', $lang)->sub_title ?? '';
        $this->return['account_tree']   = AccountTree::active_account_trees($lang , array('account_type' => 2, 'active' => 1));
        $this->return['tax_types']      = DB::table('tax_types')
            ->selectRaw('tax_type_id as id , name_'.$lang.' as name , tax_neutral')
            ->get();

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('03-003','add');

        $insert             = $request->only(['name_en', 'name_ar', 'tax_type_id' , 'details', 'tax_percentage' , 'account_tree_id']);
        $insert['add_user'] = auth()->user()->id;

        $validator = validator()->make($insert, Tax::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        $this->return['id'] = Tax::add_tax($insert);
        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('03-003','view');

        $this->return['screen_name']    = ScreenSub::get_screen_sub_by_screen_code('03-003', $lang)->sub_title ?? '';
        $this->return['account_tree']   = AccountTree::active_account_trees($lang , array('account_type' => 2, 'active' => 1));
        $this->return['data']           = Tax::get_tax($id, $lang);
        $this->return['tax_types']      = DB::table('tax_types')
            ->selectRaw('tax_type_id as id , name_'.$lang.' as name , tax_neutral')
            ->get();

        return $this->sendResponse($this->return);
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('03-003','edit');

        if(!(new TaxHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.update_fail"));
        }

        $update    = $request->only(['name_en', 'name_ar', 'tax_type_id' ,'details', 'tax_percentage' , 'account_tree_id']);
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('taxes')->ignore($id,'tax_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('taxes')->ignore($id,'tax_id'),
            ]
        ];
        $validator = validator()->make($update, $update_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        Tax::update_tax($update, $id);
        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('03-003','delete');

        if(!(new TaxHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        Tax::delete_tax($id);
        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = Tax::find($id);

        if($item->tax_active){
            $this->checkPermission('03-003','delete');

            if(!(new TaxHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            Tax::delete_tax($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('03-003','edit');

            if(!(new TaxHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            Tax::update_tax(array('tax_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
