<?php

namespace App\Http\Controllers\API\V1\FinancialStatements;

use App\Classes\Finance\VatsHelper;
use App\Http\Controllers\API\V1\BaseController;
use Illuminate\Http\Request;
use App\Models\Main\ScreenSub;
use App\Models\FinancialStatements\FinancialStatements;
use App\Models\Finance\AccountTree;
use App\Models\FinancialStatements\FinancialCollections;
use App\Models\Main\FinanceYear;
use Illuminate\Validation\Rule;
use App\Models\Finance\DailyJournal;

use DB;
class FinancialStatementsController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang , Request $request)
    {
        $this->checkPermission('02-017','view');
        $this->filter = array(
            'lang'        => $lang ,
            'active'      => $request['active'] ?? 1 ,
            'rows'        => $request['rows'] ?? null ,
            'page'        => $request['page'] ?? 1 ,
            'word'        => $request['word'] ?? null
        );
        $this->return['screen_name']    = ScreenSub::get_screen_sub_by_screen_code('02-017' , $lang)->sub_title ?? '';
        $this->return['items']          = FinancialStatements::financial_statements($this->filter);
        $this->return['account_trees']  = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));
        return $this->sendResponse($this->return);
    }

    public function create($lang , Request $request)
    {
        $this->checkPermission('02-017','add');

        $this->return['screen_name']    = ScreenSub::get_screen_sub_by_screen_code('02-017' , $lang)->sub_title ?? '';
        return $this->sendResponse( $this->return);
    }

    public function store(Request $request)
    {
        $this->checkPermission('02-017','add');

        $input    = $request->all();
        $insert = [
            'name_en'           => $input['name_en'],
            'name_ar'           => $input['name_ar'],
            'add_user'          => auth()->user()->id,
        ];

        $validator = validator()->make($insert , FinancialStatements::$rules);
        if($validator->fails())
        {
            return $this->sendError( trans('error.add_fails') , $validator->errors() );
        }
        $this->return['id'] = FinancialStatements::add_financial_statement($insert);

        return $this->sendResponse( $this->return , trans('main.add_success') );
    }

    public function show($lang , $id )
    {
        $this->checkPermission('02-017','view');
        $this->filter = array(
            'lang'          => $lang ,
            'active'        => $request['active'] ?? 1 ,
            'statement_id'  => $id,
        );
        $this->return['screen_name']     = ScreenSub::get_screen_sub_by_screen_code('02-017' , $lang)->sub_title ?? '';
        $this->return['item']            = FinancialStatements::get_financial_statement($id, $lang);
        $this->return['collections']     = FinancialCollections::all_financial_collections($this->filter);
        return $this->sendResponse( $this->return , '' );
    }
    public function get_create_report($lang )
    {
        $this->checkPermission('02-018','view');

        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('02-018' , $lang)->sub_title ?? '';
        $this->return['items']         = FinancialStatements::financial_statements(['lang'=>$lang,'active'=>1,'lookup'=>1]);
        $this->return['finance_years'] = FinanceYear::all_finance_years(array('lang' => $lang , 'active' => '1' , 'user' => auth()->user()->id ));
        return $this->sendResponse( $this->return , '' );
    }

    public function get_report($lang , Request $request )
    {
        $this->checkPermission('02-018','view');
        $branch_id          = $this->data['current_branch'] ?? 0;
        if( $branch_id == 0 ){
            return $this->sendError( 'Shift is not started' , [] , 401  );
        }
        $this->filter =[
            'lang'          => $lang,
            'active'        => 1,
            'statement_id'  => $request['statement_id'],
            'months'        => explode(',',$request['months']),
            'years'         => explode(',',$request['finance_years']),
            'branch_id'     => $branch_id,
        ];
        $this->return['screen_name']  = ScreenSub::get_screen_sub_by_screen_code('02-018' , $lang)->sub_title ?? '';
        $this->return['collections']  = FinancialCollections::all_financial_collections($this->filter);
        $this->return['report']       = [];

        // return $this->return ;

        $accounts = DB::table('financial_collections')
            ->leftjoin('financial_statements','financial_statements.statement_id', '=' ,'financial_collections.statement_id')
            ->leftjoin('account_tree','financial_collections.account_id' , '=' , 'account_tree.account_tree_id')
            ->where('financial_collections.type',2)
            ->where('financial_collections.statement_id',$this->filter['statement_id'])
            ->where('financial_statements.active' , 1)
            ->orderBy('financial_collections.collection_id','desc')
            ->selectRaw('
                financial_collections.collection_id,
                financial_collections.statement_id,
                financial_collections.account_id ,
                financial_collections.name_'.$lang.' as name ,
                financial_collections.factor,
                financial_collections.parent_id,
                account_tree.name_'.$lang.' as account_name ,
                financial_collections.type ,
                0 as amount ,
                balance_type
            ')
            ->get();

        $years = explode(',',$request['finance_years']);

        foreach($years as $key => $year) {
            foreach($accounts as $account) {

                $line           = (array) $account;
                $line['year']   = $year;
                $line['month']  = $this->filter['months'][0];

                switch ($account->balance_type) {
                    case '1':
                        $date   = date("Y-m-d" , strtotime( $year."-".$this->filter['months'][0]."-01"));
                        $line['amount'] = DailyJournal::get_account_balance($line['account_id'] , $lang , array('total' => true , 'before' => $date))->total;
                    break;
                    case '2':
                        $from   = date("Y-m-d" , strtotime( $year."-".$this->filter['months'][0]."-01"));
                        $to     = date("Y-m-t" , strtotime($year."-".$this->filter['months'][array_key_last($this->filter['months'])]."-01"));
                        $line['amount'] = DailyJournal::get_account_balance($line['account_id'] , $lang , array('total' => true , 'from' => $from , 'to' => $from  ) )->total ;
                    break;
                    case '3':
                        $date     = date("Y-m-t" , strtotime($year."-".$this->filter['months'][array_key_last($this->filter['months'])]."-01"));
                        $date = date("Y-m-d" , strtotime($date." +1 days"));

                        $line['amount'] = DailyJournal::get_account_balance($line['account_id'] , $lang , array('total' => true , 'before' => $date))->total;
                    break;
                }

                $this->return['report'][] = $line;
            }
        }

        $this->return['report'] = collect($this->return['report'])->groupBy(['year','month'])->all();

        // $this->return['report']       = FinancialStatements::get_report($this->filter);

        return $this->sendResponse( $this->return , '' );
    }

    public function update($lang, $id , Request $request)
    {
        $this->checkPermission('02-017','edit');

        $input  = $request->all();
        $update = [
            'name_en'           => $input['name_en'],
            'name_ar'           => $input['name_ar'],
        ];
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('financial_statements')->ignore($id,'statement_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('financial_statements')->ignore($id,'statement_id'),
            ],
        ];
        $validator  = validator()->make( $update , $update_rules);
        if($validator->fails())
        {
            return $this->sendError( trans('error.add_fails') , $validator->errors() );
        }

        FinancialStatements::update_financial_statement($update ,$id );
        return $this->sendResponse( [] , trans('main.update_success') );
    }

    public function destroy($lang,$id)
    {
        $this->checkPermission('02-017','delete');

        FinancialStatements::delete_financial_statement($id);

        return $this->sendResponse( [] , trans('main.delete_success') );
    }


    public function toggle_active($lang,$id){
        $item = FinancialStatements::get_financial_statement($id,$lang);
        if($item->active){

            $this->checkPermission('02-017','delete');

            FinancialStatements::delete_financial_statement($id);
            $this->return['active'] = false;

        }else{

            $this->checkPermission('02-017','edit');

            FinancialStatements::update_financial_statement(array('active' => 1 ) , $id);
            $this->return['active'] = true;
        }
        return $this->sendResponse( $this->return , trans('main.update_success') );
    }

}
