<?php

namespace App\Http\Controllers\API\V1\Inventory;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Inventory\UnitMeasurement;
use Illuminate\Http\Request;
use App\Classes\Inventory\UnitMeasurementHelper;
use App\Models\Main\ScreenSub;
use App\Models\Finance\AccountTree;
use Illuminate\Validation\Rule;

class UnitMeasurementController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('03-005','view');

        $this->filter = array(
            'lang'   => $lang ,
            'active' => $request['active'] ?? 1 ,
            'rows'   => $request['rows'] ?? null ,
            'page'   => $request['page'] ?? 1 ,
            'word'   => $request['word'] ?? null ,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-005', $lang)->sub_title ?? '';
        $this->return['items']       = UnitMeasurement::all_unit_measurements($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('03-005','add');

        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('03-005', $lang)->sub_title ?? '';
        $this->return['account_trees'] = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('03-005','add');

        $insert                    = $request->only(['name_en', 'name_ar', 'account_tree_id']);
        $insert['add_user']        = auth()->user()->id;

        $validator = validator()->make($insert, UnitMeasurement::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        $this->return['id'] = UnitMeasurement::add_unit_measurement($insert);

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('03-005','view');

        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('03-005', $lang)->sub_title ?? '';
        $this->return['data']          = UnitMeasurement::get_unit_measurement($id, $lang);
        $this->return['account_trees'] = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));

        return $this->sendResponse($this->return);
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('03-005','edit');

        if( !(new UnitMeasurementHelper)->can_edit_or_delete($id) ){
            return $this->sendError(trans("error.update_fail"));
        }

        $update = $request->only(['name_en', 'name_ar', 'account_tree_id']);
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('unit_measurements')->ignore($id,'unit_measurement_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('unit_measurements')->ignore($id,'unit_measurement_id'),
            ]
        ];
        $validator  = validator()->make($update, $update_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        UnitMeasurement::update_unit_measurement($update, $id);

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('03-005','delete');

        if(!(new UnitMeasurementHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        UnitMeasurement::delete_unit_measurement($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = UnitMeasurement::find($id);

        if($item->unit_measurement_active){
            $this->checkPermission('03-005','delete');

            if(!(new UnitMeasurementHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            UnitMeasurement::delete_unit_measurement($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('03-005','edit');

            if(!(new UnitMeasurementHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            UnitMeasurement::update_unit_measurement(array('unit_measurement_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
