<?php

namespace App\Http\Controllers\API\V1\Inventory;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Inventory\Warehouse;
use Illuminate\Http\Request;
use App\Classes\Inventory\WarehouseHelper;
use App\Models\Main\ScreenSub;
use App\Models\Finance\AccountTree;
use Illuminate\Validation\Rule;

class WarehouseController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('03-001','view');

        $this->filter = array(
            'lang'   => $lang ,
            'active' => $request['active'] ?? 1 ,
            'rows'   => $request['rows'] ?? null ,
            'page'   => $request['page'] ?? 1 ,
            'word'   => $request['word'] ?? null ,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-001', $lang)->sub_title ?? '';
        $this->return['items']       = Warehouse::all_warehouses($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('03-001','add');


        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('03-001', $lang)->sub_title ?? '';
        $this->return['account_trees'] = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('03-001','add');


        $insert                    = $request->only(['name_en', 'name_ar', 'account_tree_id']);
        $insert['add_user']        = auth()->user()->id;
        $insert['account_tree_id'] = (is_array($insert['account_tree_id'] ?? '')) ? implode(',', $insert['account_tree_id']) : $insert['account_tree_id'];

        $validator = validator()->make($insert, Warehouse::$rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        $this->return['id'] = Warehouse::add_warehouse($insert);

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('03-001','view');


        $this->return['screen_name']   = ScreenSub::get_screen_sub_by_screen_code('03-001', $lang)->sub_title ?? '';
        $this->return['data']          = Warehouse::get_warehouse($id, $lang);
        $this->return['account_trees'] = AccountTree::active_account_trees($lang, array('account_type' => 2, 'active' => '1'));

        return $this->sendResponse($this->return);
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('03-001','edit');


        if(!(new WarehouseHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.update_fail"));
        }

        $update                    = $request->only(['name_en', 'name_ar', 'account_tree_id']);
        $update['account_tree_id'] = (is_array($update['account_tree_id'] ?? '')) ? implode(',', $update['account_tree_id']) : $update['account_tree_id'];
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('warehouses')->ignore($id,'warehouse_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('warehouses')->ignore($id,'warehouse_id'),
            ]
        ];
        $validator  = validator()->make($update, $update_rules);

        if($validator->fails())
        {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        Warehouse::update_warehouse($update, $id);

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('03-001','delete');


        if(!(new WarehouseHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        Warehouse::delete_warehouse($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = Warehouse::find($id);

        if($item->warehouse_active){
            $this->checkPermission('03-001','delete');


            if(!(new WarehouseHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.delete_fails'), []);
            }

            Warehouse::delete_warehouse($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('03-001','edit');


            if(!(new WarehouseHelper)->can_edit_or_delete($id)){
                return $this->sendError(trans('error.update_fails'), []);
            }

            Warehouse::update_warehouse(array('warehouse_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
