<?php
namespace App\Http\Controllers\API\V1\Sales;
use App\Http\Controllers\API\V1\BaseController;

use App\Classes\Excel\InvoicesExport;
use App\Classes\Sales\QuotationHelper;
use App\Classes\Sales\SalesGeneralHelper;
use App\Classes\General\General;

use App\Models\Finance\Tax;
use App\Models\Inventory\Item;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\Branch;
use App\Models\Main\ScreenSub;
use App\Models\Main\Setting;
use App\Models\Main\Input;
use App\Models\Sales\Customer;
use App\Models\Sales\Quotation;
use App\Models\Sales\SalesAgent;
use Illuminate\Http\Request;

use Salla\ZATCA\GenerateQrCode;
use Salla\ZATCA\Tags\InvoiceDate;
use Salla\ZATCA\Tags\InvoiceTaxAmount;
use Salla\ZATCA\Tags\InvoiceTotalAmount;
use Salla\ZATCA\Tags\Seller;
use Salla\ZATCA\Tags\TaxNumber;

use DB;
use Auth;
use Str;
use App;
use App\Models\Finance\Currency;
use PDF;
use Excel;

class QuotationController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        if(!@in_array('1', $this->data['mypermissions']['06-005'][0]->permission)){
            return $this->sendError('view permission needed #06-005', $this->data, 200);
        }

        $this->filter = array(
            'lang'         => $lang ,
            'active'       => $request['active'] ?? 1 ,
            'status'       => $request['status'] ?? 1 ,
            'rows'         => $request['rows'] ?? 20 ,
            'page'         => $request['page'] ?? 1 ,
            'word'         => $request['word'] ?? null ,
            'branch'       => $this->data['current_branch'] ,
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-005', $lang)->sub_title ?? '';
        $this->return['items']       = Quotation::all_quotations($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        if(!@in_array('2', $this->data['mypermissions']['06-005'][0]->permission)){
            return $this->sendError('create permission needed #06-005', [], 200);
        }

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-005', $lang)->sub_title ?? '';
        $this->return['customers']         = Customer::all_customers(array('active' => 1, 'lang' => $lang ,  'is_search' => 1));
        $this->return['sales_agents']      = SalesAgent::all_sales_agents(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['width_height']      = boolval( Setting::get_main_setting_by_code('width_height')->value ?? false );

        $extra_inputs = (new General)->get_extra_inputs(92 , $lang);
        $this->return['master_inputs']     = $extra_inputs['master'];
        $this->return['detail_inputs']     = $extra_inputs['detail'];
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        if(!@in_array('2', $this->data['mypermissions']['06-005'][0]->permission)){
            return $this->sendError('create permission needed #06-005', [], 200);
        }

        $insert = $request->only(['quotation_date', 'quotation_description', 'customer_id', 'sales_agent_id','invoice_discount', 'discount_type', 'discount', 'items' , 'extra_inputs','validity','term_payment','currency_id']);
        $year   = array_search(date("Y", strtotime($insert['quotation_date'])), $this->data['years']);
        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        if(!isset($insert['items']) || !is_array($insert['items']) || count($insert['items']) < 1 ){
            return $this->sendError('Erro In Items');
        }

        $codes          = (new QuotationHelper)->get_next_codes($insert['quotation_date'], $year, $this->data['current_branch']);
        $master_seed    = [
            'branch_id'             => $this->data['current_branch'],
            'finance_year_id'       => $year,
            'year_ser'              => $codes['year_ser'],
            'month_ser'             => $codes['month_ser'],
            'branch_ser'            => $codes['branch_ser'],
            'extra_inputs'          => json_encode($insert['extra_inputs'] ?? [] ),
            'daily_journal_m_id'    => 0,
            'quotation_date'        => $insert['quotation_date'],
            'quotation_description' => $insert['quotation_description'],
            'customer_id'           => $insert['customer_id'],
            'sales_agent_id'        => $insert['sales_agent_id'],
            'invoice_discount'      => $insert['invoice_discount'] ?? 0,
            'discount_type'         => $insert['discount_type'] ?? 0,
            'validity'              => $insert['validity'] ?? null,
            'term_payment'          => $insert['term_payment'] ?? null,
            'discount'              => $insert['discount'] ?? 0,
            'total_discount'        =>  0 ,
            'add_user'              => auth()->user()->id,
            'currency_id'           => $insert['currency_id']
        ];

        $validator      = validator()->make($master_seed, Quotation::$master_rules);
        if($validator->fails()){
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

            $m_id   = Quotation::add_quotation_m($master_seed);
            $items  = (new QuotationHelper)->prepare_details( $m_id , $insert['items'] , 0 );


            if(count($items) < 1 ){
                DB::rollback();
                return $this->sendError(trans('error.add_fails'), []);
            }

            foreach ($items as $item){
                $validator = validator()->make($item, Quotation::$details_rules);
                if($validator->fails()){
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }
                unset($item['d_id']);
                $d_id = Quotation::add_quotation_d($item);
            }

            $updateMasterValues   = (new QuotationHelper)->update_master_values($m_id);
            $generateDailyJournal = (new QuotationHelper)->generate_daily_journal($m_id);

        DB::commit();
        return $this->sendResponse($m_id, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        if(!@in_array('1', $this->data['mypermissions']['06-005'][0]->permission)){
            return $this->sendError('show permission needed #06-005', [], 200);
        }

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-005', $lang)->sub_title ?? '';
        $this->return['master']            = Quotation::get_quotation_m($id, $lang);
        $this->return['details']           = Quotation::get_quotation_d($id, $lang);
        $this->return['customers']         = Customer::all_customers(array('active' => 1, 'lang' => $lang ,  'is_search' => 1));
        $this->return['sales_agents']      = SalesAgent::all_sales_agents(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['width_height']      = boolval( Setting::get_main_setting_by_code('width_height')->value ?? false );
        $this->return['master']->extra_inputs = json_decode($this->return['master']->extra_inputs);
        foreach ($this->return['details'] as $key => $value) {
            $this->return['details'][$key]->extra_inputs = json_decode($this->return['details'][$key]->extra_inputs);
        }

        $extra_inputs = (new General)->get_extra_inputs(92 , $lang);
        $this->return['master_inputs']     = $extra_inputs['master'];
        $this->return['detail_inputs']     = $extra_inputs['detail'];
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        if(!@in_array('3', $this->data['mypermissions']['06-005'][0]->permission)){
            return $this->sendError('update permission needed #06-005', [], 200);
        }

        if( !(new QuotationHelper)->can_edit_or_delete($id) ){
            return $this->sendError( trans('error.cant_edit') , [], 200);
        }

        $update = $request->only(['quotation_date', 'quotation_description', 'customer_id', 'sales_agent_id' , 'discount_type' , 'discount' ,'validity','term_payment', 'extra_inputs' ,'currency_id']);
        $year   = array_search(date("Y", strtotime($update['quotation_date'])), $this->data['years']);
        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        if(!isset($request['items']) || !is_array($request['items']) || count($request['items']) < 1 ){
            return $this->sendError('Erro In Items');
        }

        DB::beginTransaction();
            $update['extra_inputs'] = json_encode($update['extra_inputs'] ?? [] );
            Quotation::update_quotation_m($update, $id);

            DB::table('quotation_d')
                ->where('quotation_m_id' , $id)
                ->delete();

            $items  = (new QuotationHelper)->prepare_details( $id , $request['items'] , 0 );

            if(count($items) < 1 ){
                DB::rollback();
                return $this->sendError(trans('error.add_fails'), []);
            }

            foreach ($items as $item){

                $validator = validator()->make($item, Quotation::$details_rules);
                if($validator->fails()){
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }

                unset($item['d_id']);
                $d_id = Quotation::add_quotation_d($item);
            }


            $updateMasterValues   = (new QuotationHelper)->update_master_values($id);
            $generateDailyJournal = (new QuotationHelper)->generate_daily_journal($id);

        DB::commit();


        return $this->sendResponse($id, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        if(!@in_array('4', $this->data['mypermissions']['06-005'][0]->permission)){
            return $this->sendError('delete permission needed #06-005', [], 200);
        }

        $master = Quotation::get_quotation_m($id);

        if(!isset($master->quotation_m_id) || $master->branch_id != $this->data['current_branch'] || !(new QuotationHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        Quotation::delete_quotation_m($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $master = Quotation::find($id);
        if(!isset($master->quotation_m_id) || $master->branch_id != $this->data['current_branch'] || !(new QuotationHelper)->can_edit_or_delete($id)){
            return $this->sendError(trans("error.delete_fail"));
        }

        if($master->quotation_active){
            if(!@in_array('4', $this->data['mypermissions']['06-005'][0]->permission)){
                return $this->sendError('update permission needed #06-005', [], 200);
            }

            Quotation::delete_quotation_m($id);
            $this->return['active'] = false;
        }else{
            if(!@in_array('3', $this->data['mypermissions']['06-005'][0]->permission)){
                return $this->sendError('update permission needed #06-005', [], 200);
            }

            if(!(new QuotationHelper)->can_edit_or_delete($id)){
                return $this->sendError( trans('error.update_fails'), []);
            }

            Quotation::update_quotation_m(array('quotation_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function accept_quotation($lang, $id, Request $request)
    {
        $quotation = DB::table('quotation_m')
            ->where('quotation_m_id', $id)
            ->update([
                'quotation_status' => 3
            ]);
        return $this->sendResponse([], trans('main.Quotation Accepted Successfully'));
    }

    public function decline_quotation($lang, $id, Request $request)
    {
        $quotation = DB::table('quotation_m')
            ->where('quotation_m_id', $id)
            ->update([
                'quotation_status' => 2
            ]);
        return $this->sendResponse([], trans('main.Quotation Declined Successfully'));
    }

    public function print_invoice($id, Request $request)
    {
        $input = $request->all();
        App::setLocale('ar');

        $this->data['lang']        = 'ar';
        $this->data['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-008' , $this->data['lang']);
        $this->data['master']      = Quotation::get_quotation_m($id, $this->data['lang']);
        $this->data['details']     = Quotation::get_quotation_d($id, $this->data['lang']);
        $this->data['customer']    = Customer::get_customer($this->data['master']->customer_id, $this->data['lang']);
        $this->data['logo']        = get_logo($this->data['master']->branch_id);
        $this->data['branch_data'] = Branch::get_branch($this->data['master']->branch_id);

        $temp_name                 = $input['print_temp'] ?? Setting::get_main_setting_by_code('quotation_print')->value ?? 'temp_1';
        $print_temp                = 'Print.Sales.quotation.'.$temp_name;

        $this->data['qrcode']      = GenerateQrCode::fromArray([
            new Seller($this->data['branch_data']->vat_register_name), // Seller Name
            new TaxNumber($this->data['branch_data']->vat_number), // Seller Tax Number
            new InvoiceDate(date('Y-m-d\TH:i:s', strtotime($this->data['master']->add_date))),
            new InvoiceTotalAmount($this->data['master']->total_with_vat), // Invoice Total Amount
            new InvoiceTaxAmount(round($this->data['master']->total_vat, 2)) // Invoice Tax Amount
        ])->render();

        switch (@$input['type']) {
            case 'excel':
                $this->data['logo'] = public_path().$this->data['logo'];
                $export = new InvoicesExport($this->data, $print_temp);
                return Excel::download($export, 'balances.xlsx');
                break;
            case 'pdf':
                $this->data['logo'] = public_path().$this->data['logo'];
                $pdf = PDF::loadView($print_temp, array('data' => $this->data ));
                return $pdf->stream();
                break;
            case 'image':
                $this->data['logo'] = public_path().$this->data['logo'];
                $image = App::make('snappy.image.wrapper');
                $image->loadView($print_temp, array('data' => $this->data));
                return $image->stream();
                break;
            case 'print':
                return view($print_temp)->with('data', $this->data);
                break;
        }
    }
}
